/**
 *  Copyright (c) 2012 BMW
 *
 *  \author Christian Mueller, christian.ei.mueller@bmw.de BMW 2011,2012
 *
 *  \copyright
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 *  subject to the following conditions:
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 *  THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  For further information see http://www.genivi.org/.
 */


/************************************************************************
 *FILE:      IAmRoutingReceiverShadow.cpp
 *SW-COMPONENT:    G3G Audio Component
 *DESCRIPTION:    Wraps the IAmRoutingReceive interface and serializes the access to these interfaces.

 *AUTHOR:    RBEI/ECG1 - Bipin Krishnan
 *COPYRIGHT:
 *HISTORY:
 *06.09.2012    Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/



#include "IAmRoutingReceiverShadow.h"
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <string.h>
#include <netdb.h>
#include <fcntl.h>
#include <sys/un.h>
#include <errno.h>
#include <string>
#include "CAmDltWrapper.h"
#include "CAmSerializer.h"

using namespace am;


/************************************************************************
 *FUNCTION:  IAmRoutingReceiverShadow
 *DESCRIPTION:  Constructor to initialize IAmRoutingReceiverShadow object.
 *PARAMETER:
    iReceiveInterface: (->I) Pointer to IAmRoutingReceive
    iSocketHandler: (->I) Pointer to CAmSocketHandler

 *RETURN VALUE: NULL
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

IAmRoutingReceiverShadow::IAmRoutingReceiverShadow(IAmRoutingReceive* iReceiveInterface, CAmSocketHandler* iSocketHandler) :
        mSocketHandler(iSocketHandler), //
        mRoutingReceiveInterface(iReceiveInterface), //
        mSerializer(iSocketHandler)
{

}


/************************************************************************
 *FUNCTION:  ~IAmRoutingReceiverShadow
 *DESCRIPTION:  Destructor of IAmRoutingReceiverShadow.
 *PARAMETER:
 *RETURN VALUE: NULL
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

IAmRoutingReceiverShadow::~IAmRoutingReceiverShadow()
{
  mSocketHandler = 0;
  mRoutingReceiveInterface = 0;
}


/************************************************************************
 *FUNCTION:  ackConnect
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::ackConnect.
 *PARAMETER:
    handle: (I) handle of acknowledgement of connect request.
    connectionID: (I) connection identifier.
    error: (I) error number.

 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/
void IAmRoutingReceiverShadow::ackConnect(const am_Handle_s handle, const am_connectionID_t connectionID, const am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_connectionID_t, const am_Error_e>(mRoutingReceiveInterface, &IAmRoutingReceive::ackConnect, handle, connectionID, error);
}


/************************************************************************
 *FUNCTION:  ackDisconnect
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::ackDisconnect.
 *PARAMETER:
    handle: (I) handle of the acknowledgement of disconnect request.
    connectionID: (I) connection identifier.
    error: (I) error number.
 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

void IAmRoutingReceiverShadow::ackDisconnect(const am_Handle_s handle, const am_connectionID_t connectionID, const am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_connectionID_t, const am_Error_e>(mRoutingReceiveInterface, &IAmRoutingReceive::ackDisconnect, handle, connectionID, error);
}


/************************************************************************
 *FUNCTION:  ackSetSinkVolumeChange
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::ackSetSinkVolumeChange.

 *PARAMETER:
    handle: (I) handle of the acknowledgement of Volume Change - Sink request.
    volume: (I) volume of the sink.
    error: (I) error number.
 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

void IAmRoutingReceiverShadow::ackSetSinkVolumeChange(const am_Handle_s handle, const am_volume_t volume, const am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_volume_t, const am_Error_e>(mRoutingReceiveInterface, &IAmRoutingReceive::ackSetSinkVolumeChange, handle, volume, error);
}


/************************************************************************
 *FUNCTION:  ackSetSourceVolumeChange
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive:: ackSetSourceVolumeChange.
 *PARAMETER:
    handle: (I) handle of the acknowledgement of Volume Change - Source request.
    volume: (I) volume of the source.
    error: (I) error number.
 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

void IAmRoutingReceiverShadow::ackSetSourceVolumeChange(const am_Handle_s handle, const am_volume_t volume, const am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_volume_t, const am_Error_e>(mRoutingReceiveInterface, &IAmRoutingReceive::ackSetSourceVolumeChange, handle, volume, error);
}

void IAmRoutingReceiverShadow::ackSetSourceState(const am_Handle_s handle, const am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_Error_e>(mRoutingReceiveInterface, &IAmRoutingReceive::ackSetSourceState, handle, error);
}

void IAmRoutingReceiverShadow::ackSetSinkSoundProperty(const am_Handle_s handle, const am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_Error_e>(mRoutingReceiveInterface, &IAmRoutingReceive::ackSetSinkSoundProperty, handle, error);
}
void IAmRoutingReceiverShadow::ackSetSinkSoundProperties(const am_Handle_s handle, const am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_Error_e>(mRoutingReceiveInterface, &IAmRoutingReceive::ackSetSinkSoundProperties, handle, error);
}
void IAmRoutingReceiverShadow::ackSetSourceSoundProperty(const am_Handle_s handle, const am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_Error_e>(mRoutingReceiveInterface, &IAmRoutingReceive::ackSetSourceSoundProperty, handle, error);
}
void IAmRoutingReceiverShadow::ackSetSourceSoundProperties(const am_Handle_s handle, const am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_Error_e>(mRoutingReceiveInterface, &IAmRoutingReceive::ackSetSourceSoundProperties, handle, error);
}
void IAmRoutingReceiverShadow::ackCrossFading(const am_Handle_s handle, const am_HotSink_e hotSink, const am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_HotSink_e, const am_Error_e>(mRoutingReceiveInterface, &IAmRoutingReceive::ackCrossFading, handle, hotSink, error);
}

void IAmRoutingReceiverShadow::ackSourceVolumeTick(const am_Handle_s handle, const am_sourceID_t sourceID, const am_volume_t volume)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_sourceID_t, const am_volume_t>(mRoutingReceiveInterface, &IAmRoutingReceive::ackSourceVolumeTick, handle, sourceID, volume);
}

void IAmRoutingReceiverShadow::ackSinkVolumeTick(const am_Handle_s handle, const am_sinkID_t sinkID, const am_volume_t volume)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_Handle_s, const am_sinkID_t, const am_volume_t>(mRoutingReceiveInterface, &IAmRoutingReceive::ackSinkVolumeTick, handle, sinkID, volume);
}

void IAmRoutingReceiverShadow::hookInterruptStatusChange(const am_sourceID_t sourceID, const am_InterruptState_e interruptState)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_sinkID_t, const am_InterruptState_e>(mRoutingReceiveInterface, &IAmRoutingReceive::hookInterruptStatusChange, sourceID, interruptState);
}

void IAmRoutingReceiverShadow::hookSinkAvailablityStatusChange(const am_sinkID_t sinkID, const am_Availability_s & availability)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_sinkID_t, const am_Availability_s>(mRoutingReceiveInterface, &IAmRoutingReceive::hookSinkAvailablityStatusChange, sinkID, availability);
}

void IAmRoutingReceiverShadow::hookSourceAvailablityStatusChange(const am_sourceID_t sourceID, const am_Availability_s & availability)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_sourceID_t, const am_Availability_s>(mRoutingReceiveInterface, &IAmRoutingReceive::hookSourceAvailablityStatusChange, sourceID, availability);
}

void IAmRoutingReceiverShadow::hookDomainStateChange(const am_domainID_t domainID, const am_DomainState_e domainState)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_domainID_t, const am_DomainState_e>(mRoutingReceiveInterface, &IAmRoutingReceive::hookDomainStateChange, domainID, domainState);
}


/************************************************************************
 *FUNCTION:  hookTimingInformationChanged
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive:: hookTimingInformationChanged.

 *PARAMETER:   connectionID: (I) connection identifier.
    delay: (I) time delay to establish the connection.

 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

void IAmRoutingReceiverShadow::hookTimingInformationChanged(const am_connectionID_t connectionID, const am_timeSync_t delay)
{
    mSerializer.asyncCall<IAmRoutingReceive, const am_connectionID_t, const am_timeSync_t>(mRoutingReceiveInterface, &IAmRoutingReceive::hookTimingInformationChanged, connectionID, delay);
}


/************************************************************************
 *FUNCTION:  registerDomain
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::registerDomain.
 *PARAMETER:   domainData: (IO) domain information.
    domainID: (IO) domain identifier.

 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

am_Error_e IAmRoutingReceiverShadow::registerDomain(const am_Domain_s & domainData, am_domainID_t & domainID)
{
    am_Error_e error (E_UNKNOWN);
    am_Domain_s domainDataCopy(domainData);
    //mSerializer.syncCall<IAmRoutingReceive, am_Error_e, const am_Domain_s&,am_domainID_t&, am_Domain_s, am_domainID_t>(mRoutingReceiveInterface, &IAmRoutingReceive::registerDomain, error, domainDataCopy, domainID);
  error = mRoutingReceiveInterface->registerDomain(domainDataCopy,domainID);
    return (error);
}

/************************************************************************
 *FUNCTION:  deregisterDomain
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::deregisterDomain.
 *PARAMETER:  domainID: (IO) domain identifier.

 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/
am_Error_e IAmRoutingReceiverShadow::deregisterDomain(const am_domainID_t domainID)
{
  am_Error_e error (E_UNKNOWN);
    am_domainID_t domID(domainID);
    //mSerializer.syncCall<IAmRoutingReceive, am_Error_e, am_domainID_t>(mRoutingReceiveInterface, &IAmRoutingReceive::deregisterDomain, error,domID);
  error = mRoutingReceiveInterface->deregisterDomain(domID);
    return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::registerGateway(const am_Gateway_s & gatewayData, am_gatewayID_t & gatewayID)
{
    am_Error_e error (E_UNKNOWN);
    am_Gateway_s gatewayDataCopy(gatewayData);
    //mSerializer.syncCall<IAmRoutingReceive, am_Error_e, const am_Gateway_s&, am_gatewayID_t&, am_Gateway_s, am_gatewayID_t>(mRoutingReceiveInterface,&IAmRoutingReceive::registerGateway, error, gatewayDataCopy, gatewayID);
  error = mRoutingReceiveInterface->registerGateway(gatewayDataCopy, gatewayID);
    return (error);
}


/************************************************************************
 *FUNCTION:  registerSink
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::registerSink.
 *PARAMETER:
    sinkData: (IO) sink information.
    sinkID: (IO) sink identifier.
 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

am_Error_e am::IAmRoutingReceiverShadow::registerSink(const am_Sink_s & sinkData, am_sinkID_t & sinkID)
{
    am_Error_e error (E_UNKNOWN);
    am_Sink_s sinkDataCopy(sinkData);
    //mSerializer.syncCall<IAmRoutingReceive, am_Error_e, const am_Sink_s&, am_sinkID_t&, am_Sink_s, am_sinkID_t>(mRoutingReceiveInterface,&IAmRoutingReceive::registerSink, error, sinkDataCopy, sinkID);
  error = mRoutingReceiveInterface->registerSink(sinkDataCopy, sinkID);
    return (error);
}


/************************************************************************
 *FUNCTION:  deregisterSink
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::deregisterSink.
 *PARAMETER:   sinkID: (I) sink identifier.

 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

am_Error_e am::IAmRoutingReceiverShadow::deregisterSink(const am_sinkID_t sinkID)
{
    am_Error_e error;
    am_sinkID_t s(sinkID); //no const values allowed in syncCalls due to reference !
    //mSerializer.syncCall<IAmRoutingReceive, am_Error_e, am_sinkID_t>(mRoutingReceiveInterface, &IAmRoutingReceive::deregisterSink, error, s);
  error = mRoutingReceiveInterface->deregisterSink(s);
    return (error);
}


/************************************************************************
 *FUNCTION:  registerSource
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::registerSource.
 *PARAMETER:
    sourceData: (IO) source information.
    sourceID: (IO) source identifier.

 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

am_Error_e am::IAmRoutingReceiverShadow::registerSource(const am_Source_s & sourceData, am_sourceID_t & sourceID)
{
    am_Error_e error (E_UNKNOWN);
    am_Source_s sourceDataCopy(sourceData);
    //mSerializer.syncCall<IAmRoutingReceive, am_Error_e, const am_Source_s&, am_sourceID_t&, am_Source_s, am_sourceID_t>(mRoutingReceiveInterface,&IAmRoutingReceive::registerSource, error, sourceDataCopy, sourceID);
  error = mRoutingReceiveInterface->registerSource(sourceDataCopy, sourceID);
    return (error);
}


/************************************************************************
 *FUNCTION:  deregisterSource
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::deregisterSource.
 *PARAMETER:   sourceID: (I) source identifier.

 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

am_Error_e am::IAmRoutingReceiverShadow::deregisterSource(const am_sourceID_t sourceID)
{
    am_Error_e error;
    am_sourceID_t s(sourceID); //no const values allowed in syncCalls due to reference !
    //mSerializer.syncCall<IAmRoutingReceive, am_Error_e, am_sinkID_t>(mRoutingReceiveInterface, &IAmRoutingReceive::deregisterSource, error, s);
  error = mRoutingReceiveInterface->deregisterSource(s);
    return (error);
}

am_Error_e am::IAmRoutingReceiverShadow::registerCrossfader(const am_Crossfader_s & crossfaderData, am_crossfaderID_t & crossfaderID)
{
    am_Error_e error (E_UNKNOWN);
    am_Crossfader_s crossfaderDataCopy(crossfaderData);
    mSerializer.syncCall<IAmRoutingReceive, am_Error_e, const am_Crossfader_s&, am_crossfaderID_t&, am_Crossfader_s, am_crossfaderID_t>(mRoutingReceiveInterface,&IAmRoutingReceive::registerCrossfader, error, crossfaderDataCopy, crossfaderID);
    return (error);
}


/************************************************************************
 *FUNCTION:  confirmRoutingReady
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::confirmRoutingReady.
 *PARAMETER:
    startupHandle: (I) handle of acknowledgement of the startup request.

 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/
//modified
void am::IAmRoutingReceiverShadow::confirmRoutingReady(uint16_t startupHandle, am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive,uint16_t,am_Error_e>(mRoutingReceiveInterface,&IAmRoutingReceive::confirmRoutingReady,startupHandle,error);
}


/************************************************************************
 *FUNCTION:  confirmRoutingRundown
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::confirmRoutingRundown.
 *PARAMETER:
    rundownHandle: (I) handle of acknowledgement of the rundown request.

 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/
//modified
void am::IAmRoutingReceiverShadow::confirmRoutingRundown(uint16_t rundownHandle, am_Error_e error)
{
    mSerializer.asyncCall<IAmRoutingReceive,uint16_t,am_Error_e>(mRoutingReceiveInterface,&IAmRoutingReceive::confirmRoutingRundown,rundownHandle,error);
}


/************************************************************************
 *FUNCTION:  hookDomainRegistrationComplete
 *DESCRIPTION:  This method serializes the call to IAmRoutingReceive::hookDomainRegistrationComplete.
 *PARAMETER:
    domainID: (I) domain identifier.

 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG1 - Bipin Krishnan  Initial Revision
************************************************************************/

void am::IAmRoutingReceiverShadow::hookDomainRegistrationComplete(const am_domainID_t domainID)
{
    mSerializer.asyncCall<IAmRoutingReceive,const am_domainID_t>(mRoutingReceiveInterface,&IAmRoutingReceive::hookDomainRegistrationComplete,domainID);
}

am_Error_e am::IAmRoutingReceiverShadow::peekDomain(const std::string& name, am_domainID_t& domainID)
{
    am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->peekDomain(name,domainID);
    return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::registerConverter(const am_Converter_s& converterData, am_converterID_t& converterID)
{
    am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->registerConverter(converterData,converterID);
    return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::deregisterGateway(const am_gatewayID_t gatewayID)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->deregisterGateway(gatewayID);
    return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::deregisterConverter(const am_converterID_t converterID)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->deregisterConverter(converterID);
    return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::peekSink(const std::string& name, am_sinkID_t& sinkID)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->peekSink(name,sinkID);
    return (error);
}

am_Error_e am::IAmRoutingReceiverShadow::peekSource(const std::string& name, am_sourceID_t& sourceID)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->peekSource(name,sourceID);
    return (error);
}

am_Error_e am::IAmRoutingReceiverShadow::deregisterCrossfader(const am_crossfaderID_t crossfaderID)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->deregisterCrossfader(crossfaderID);
    return (error);
}

am_Error_e am::IAmRoutingReceiverShadow::peekSourceClassID(const std::string& name, am_sourceClass_t& sourceClassID)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->peekSourceClassID(name,sourceClassID);
    return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::peekSinkClassID(const std::string& name, am_sinkClass_t& sinkClassID)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->peekSinkClassID(name,sinkClassID);
    return (error);
}
void am::IAmRoutingReceiverShadow::sendChangedData(const std::vector<am_EarlyData_s>& earlyData)
{
  mRoutingReceiveInterface->sendChangedData(earlyData);
}
am_Error_e am::IAmRoutingReceiverShadow::getDBusConnectionWrapper(CAmDbusWrapper*& dbusConnectionWrapper) const
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->getDBusConnectionWrapper(dbusConnectionWrapper);
    return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::getSocketHandler(CAmSocketHandler*& socketHandler) const
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->getSocketHandler(socketHandler);
    return (error);
}
void am::IAmRoutingReceiverShadow::getInterfaceVersion(std::string& version) const
{
  mRoutingReceiveInterface->getInterfaceVersion(version);
}
am_Error_e am::IAmRoutingReceiverShadow::updateGateway(const am_gatewayID_t gatewayID, const std::vector<am_CustomConnectionFormat_t>& listSourceFormats, const std::vector<am_CustomConnectionFormat_t>& listSinkFormats, const std::vector<bool>& convertionMatrix)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->updateGateway(gatewayID,listSourceFormats,listSinkFormats,convertionMatrix);
    return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::updateConverter(const am_converterID_t converterID, const std::vector<am_CustomConnectionFormat_t>& listSourceFormats, const std::vector<am_CustomConnectionFormat_t>& listSinkFormats, const std::vector<bool>& convertionMatrix)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->updateConverter(converterID,listSourceFormats,listSinkFormats,convertionMatrix);
    return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::updateSink(const am_sinkID_t sinkID, const am_sinkClass_t sinkClassID, const std::vector<am_SoundProperty_s>& listSoundProperties, const std::vector<am_CustomConnectionFormat_t>& listConnectionFormats, const std::vector<am_MainSoundProperty_s>& listMainSoundProperties)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->updateSink(sinkID,sinkClassID,listSoundProperties,listConnectionFormats,listMainSoundProperties);
    return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::updateSource(const am_sourceID_t sourceID, const am_sourceClass_t sourceClassID, const std::vector<am_SoundProperty_s>& listSoundProperties, const std::vector<am_CustomConnectionFormat_t>& listConnectionFormats, const std::vector<am_MainSoundProperty_s>& listMainSoundProperties)
{
  am_Error_e error (E_UNKNOWN);
    error = mRoutingReceiveInterface->updateSource(sourceID,sourceClassID,listSoundProperties,listConnectionFormats,listMainSoundProperties);
    return (error);
}
void  am::IAmRoutingReceiverShadow::ackSetVolumes(const am_Handle_s handle, const std::vector<am_Volumes_s>& listvolumes, const am_Error_e error)
{
  mRoutingReceiveInterface->ackSetVolumes(handle,listvolumes,error);
}
void am::IAmRoutingReceiverShadow::ackSinkNotificationConfiguration(const am_Handle_s handle, const am_Error_e error)
{
  mRoutingReceiveInterface->ackSinkNotificationConfiguration(handle,error);
}
void am::IAmRoutingReceiverShadow::ackSourceNotificationConfiguration(const am_Handle_s handle, const am_Error_e error)
{
  mRoutingReceiveInterface->ackSourceNotificationConfiguration(handle,error);
}
void am::IAmRoutingReceiverShadow::hookSinkNotificationDataChange(const am_sinkID_t sinkID, const am_NotificationPayload_s& payload)
{
  mRoutingReceiveInterface->hookSinkNotificationDataChange(sinkID,payload);
}
void am::IAmRoutingReceiverShadow::hookSourceNotificationDataChange(const am_sourceID_t sourceID, const am_NotificationPayload_s& payload)
{
    mRoutingReceiveInterface->hookSourceNotificationDataChange(sourceID,payload);
}
am_Error_e am::IAmRoutingReceiverShadow::getDomainOfSink(const am_sinkID_t sinkID, am_domainID_t& domainID) const
{
  am_Error_e error (E_UNKNOWN);
  mRoutingReceiveInterface->getDomainOfSink(sinkID,domainID);
  return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::getDomainOfSource(const am_sourceID_t sourceID, am_domainID_t& domainID) const
{
  am_Error_e error (E_UNKNOWN);
  mRoutingReceiveInterface->getDomainOfSource(sourceID,domainID);
  return (error);
}
am_Error_e am::IAmRoutingReceiverShadow::getDomainOfCrossfader(const am_crossfaderID_t crossfader, am_domainID_t& domainID) const
{
  am_Error_e error (E_UNKNOWN);
  mRoutingReceiveInterface->getDomainOfCrossfader(crossfader,domainID);
  return (error);
}
